bl_info = {
    "name": "SinterForge",
    "author": "Sascha aka 6runk",
    "version": (0, 3, 6),
    "blender": (3, 0, 0),
    "location": "View3D > Sidebar > SinterForge",
    "description": "Apply Sintered textures with Save/Load Presets.",
    "doc_url": "https://github.com/6runk/sinter-forge", 
    "tracker_url": "https://github.com/6runk/sinter-forge/issues",
    "category": "Object",
}

import bpy
import json
import os
import glob

# --- CORE LOGIC ---

def apply_texture(obj, props):
    """Applies the texture modifiers based on UI settings."""
    bpy.ops.object.mode_set(mode='OBJECT')
    
    # 1. Scale Fix
    if props.fix_scale:
        obj.scale = (0.001, 0.001, 0.001)
        bpy.ops.object.transform_apply(location=False, rotation=False, scale=True)

    # 2. FAIL-SAFE TEXTURE CREATION
    tex_name = "SF_Texture"
    target_type = props.texture_type
    texture = None

    # Step A: Check if it exists
    if tex_name in bpy.data.textures:
        existing_tex = bpy.data.textures[tex_name]
        # Step B: Check if it's the right type
        if existing_tex.type == target_type:
            texture = existing_tex
        else:
            # Wrong type? Delete it.
            bpy.data.textures.remove(existing_tex)
            texture = bpy.data.textures.new(tex_name, type=target_type)
    else:
        # Doesn't exist? Create it.
        texture = bpy.data.textures.new(tex_name, type=target_type)
    
    # 3. APPLY SETTINGS
    if target_type == 'CLOUDS':
        texture.noise_scale = props.noise_scale
        texture.noise_depth = props.noise_detail
        
    elif target_type == 'MUSGRAVE':
        texture.musgrave_type = 'FBM' 
        texture.noise_scale = props.noise_scale
        # Clamp octaves (Musgrave limit is usually 8)
        texture.octaves = min(props.noise_detail + 1.0, 8.0) 
        
    elif target_type == 'VORONOI':
        texture.noise_scale = props.noise_scale
        texture.noise_intensity = 1.0 

    # 4. ADD MODIFIERS
    for mod in obj.modifiers:
        if mod.name.startswith("SF_"):
            obj.modifiers.remove(mod)

    # Method Selection
    if props.method == 'MECHANICAL':
        mod_sub = obj.modifiers.new(name="SF_Subdiv", type='SUBSURF')
        mod_sub.subdivision_type = 'SIMPLE' 
        mod_sub.levels = props.subdiv_levels
    elif props.method == 'ORGANIC':
        mod_remesh = obj.modifiers.new(name="SF_Remesh", type='REMESH')
        mod_remesh.mode = 'VOXEL'
        mod_remesh.voxel_size = props.voxel_size

    # Displace
    mod_disp = obj.modifiers.new(name="SF_Displace", type='DISPLACE')
    mod_disp.texture = texture
    mod_disp.strength = props.strength
    mod_disp.mid_level = 0.5

    # Decimate
    if props.use_decimate:
        mod_dec = obj.modifiers.new(name="SF_Decimate", type='DECIMATE')
        mod_dec.ratio = props.decimate_ratio

# --- PROPERTIES & SETTINGS ---

class SinterForgeSettings(bpy.types.PropertyGroup):
    method: bpy.props.EnumProperty(
        name="Method",
        description="HOW the mesh is prepared:\n• Mechanical: Safe! Keeps sharp edges (Best for Wagos/Functional Parts).\n• Organic: Destructive! Melts mesh together (Best for Minis/Terrain).",
        items=[
            ('MECHANICAL', "Mechanical (Safe)", "Uses Simple Subdivision. Keeps sharp edges. Best for Wagos/Parts."),
            ('ORGANIC', "Organic (Destructive)", "Uses Voxel Remesh. Best for Minis/Terrain/Corrosion.")
        ],
        default='MECHANICAL'
    )
    
    fix_scale: bpy.props.BoolProperty(
        name="Fix Giant Scale",
        description="Enable this ONLY if your imported STL appears 1000x too big (Meters vs Millimeters issue).",
        default=False
    )
    
    # --- TEXTURE LOOK ---
    texture_type: bpy.props.EnumProperty(
        name="Pattern",
        description="The style of the surface noise:\n• Standard Grit (Clouds): Uniform sandpaper/powder look.\n• Corrosion (Musgrave): Eroded, rusty pits.\n• Hammered (Voronoi): Chipped/dented metal.",
        items=[
            ('CLOUDS', "Standard Grit", "Uniform sandpaper/powder look."),
            ('MUSGRAVE', "Corrosion", "Eroded, rusty look."),
            ('VORONOI', "Hammered", "Chipped/Dented metal look.")
        ],
        default='CLOUDS'
    )
    
    strength: bpy.props.FloatProperty(
        name="Depth", 
        description="How deep the texture bites into the model.\n• 0.005: Fine powder (Wago rails)\n• 0.020: Heavy Rust (Terrain)\n• 0.050: Extreme Damage", 
        default=0.01, min=0.0, max=1.0, precision=3
    )
    noise_scale: bpy.props.FloatProperty(
        name="Scale", 
        description="The physical size of the individual grains.\n• 0.01: Fine sand\n• 0.05: Chunky gravel", 
        default=0.015, min=0.001, max=1.0, precision=3
    )
    noise_detail: bpy.props.IntProperty(
        name="Roughness", 
        description="Adds micro-detail complexity.\n• 0: Smooth blobs\n• 2: Standard Noise\n• 5: Extremely crunchy/fine detail",
        default=2, min=0, max=5
    )
    
    # --- RESOLUTION ---
    subdiv_levels: bpy.props.IntProperty(
        name="Subdiv Levels", 
        description="For Mechanical Mode: How many times to split faces.\n• 2: Low Detail\n• 3: Good Balance\n• 4+: Very High Detail (Warning: Slow!)", 
        default=3, min=1, max=6
    )
    voxel_size: bpy.props.FloatProperty(
        name="Voxel Size (mm)", 
        description="For Organic Mode: Resolution of the Remesh.\n• 0.03mm: Sharpest detail (Huge File)\n• 0.05mm: Standard\n• 0.10mm: Blocky/Low Poly", 
        default=0.05, min=0.01, max=1.0, precision=3
    )
    
    # --- OPTIMIZATION ---
    use_decimate: bpy.props.BoolProperty(
        name="Decimate Result", 
        description="Highly Recommended! Reduces file size by removing unnecessary geometry after texturing.",
        default=True
    )
    decimate_ratio: bpy.props.FloatProperty(
        name="Keep Ratio", 
        description="Percentage of triangles to keep.\n• 0.2: Small file (Keep 20%)\n• 0.4: Balanced (Keep 40%)\n• 0.8: High Fidelity (Keep 80%)", 
        default=0.4, min=0.01, max=1.0
    )

# --- OPERATORS ---

class SF_OT_ApplyTexture(bpy.types.Operator):
    """Apply current settings to selected objects"""
    bl_idname = "sf.apply_texture"
    bl_label = "Forge Texture"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        props = context.scene.sinter_forge_tool
        selected = context.selected_objects
        if not selected:
            self.report({'ERROR'}, "No objects selected")
            return {'CANCELLED'}
        
        wm = context.window_manager
        wm.progress_begin(0, len(selected))
        for i, obj in enumerate(selected):
            if obj.type == 'MESH':
                apply_texture(obj, props)
                wm.progress_update(i)
        wm.progress_end()
        return {'FINISHED'}

class SF_OT_SavePreset(bpy.types.Operator):
    """Save current settings as a new Preset"""
    bl_idname = "sf.save_preset"
    bl_label = "Save Preset"
    
    preset_name: bpy.props.StringProperty(name="Name", default="My Custom Preset")
    preset_desc: bpy.props.StringProperty(name="Description", default="Custom settings")

    def invoke(self, context, event):
        return context.window_manager.invoke_props_dialog(self)

    def execute(self, context):
        props = context.scene.sinter_forge_tool
        data = {
            "name": self.preset_name,
            "description": self.preset_desc,
            "method": props.method,
            "texture_type": props.texture_type,
            "strength": props.strength,
            "noise_scale": props.noise_scale,
            "noise_detail": props.noise_detail,
            "subdiv_levels": props.subdiv_levels,
            "voxel_size": props.voxel_size,
            "use_decimate": props.use_decimate,
            "decimate_ratio": props.decimate_ratio
        }
        
        filename = self.preset_name.lower().replace(" ", "_") + ".json"
        base_path = os.path.dirname(__file__)
        filepath = os.path.join(base_path, "presets", filename)
        
        try:
            if not os.path.exists(os.path.join(base_path, "presets")):
                os.makedirs(os.path.join(base_path, "presets"))
            with open(filepath, 'w') as f:
                json.dump(data, f, indent=4)
            self.report({'INFO'}, f"Saved preset: {self.preset_name}")
        except Exception as e:
            self.report({'ERROR'}, f"Failed to save: {str(e)}")
        return {'FINISHED'}

class SF_OT_LoadJSONPreset(bpy.types.Operator):
    """Load settings from a JSON file"""
    bl_idname = "sf.load_json_preset"
    bl_label = "Load Preset"
    filename: bpy.props.StringProperty()

    def execute(self, context):
        props = context.scene.sinter_forge_tool
        base_path = os.path.dirname(__file__)
        filepath = os.path.join(base_path, "presets", self.filename)
        try:
            with open(filepath, 'r') as f:
                data = json.load(f)
                if "method" in data: props.method = data["method"]
                if "texture_type" in data: props.texture_type = data.get("texture_type", 'CLOUDS')
                if "strength" in data: props.strength = data["strength"]
                if "noise_scale" in data: props.noise_scale = data["noise_scale"]
                if "noise_detail" in data: props.noise_detail = data.get("noise_detail", 2)
                if "subdiv_levels" in data: props.subdiv_levels = data["subdiv_levels"]
                if "voxel_size" in data: props.voxel_size = data["voxel_size"]
                if "use_decimate" in data: props.use_decimate = data["use_decimate"]
                if "decimate_ratio" in data: props.decimate_ratio = data["decimate_ratio"]
            self.report({'INFO'}, f"Loaded: {data.get('name', self.filename)}")
        except Exception as e:
            self.report({'ERROR'}, f"Error: {str(e)}")
        return {'FINISHED'}

# --- PREFERENCES ---

class SinterForgePreferences(bpy.types.AddonPreferences):
    bl_idname = __name__

    def draw(self, context):
        layout = self.layout
        box = layout.box()
        row = box.row()
        row.label(text="SinterForge is made possible by:", icon='HEART')
        op = row.operator("wm.url_open", text="Visit My Sponsor", icon='FUND')
        op.url = "https://aus.gmbh" 

# --- PANEL ---

class SF_PT_MainPanel(bpy.types.Panel):
    bl_label = "SinterForge"
    bl_idname = "SF_PT_MainPanel"
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'UI'
    bl_category = "SinterForge"

    def draw(self, context):
        layout = self.layout
        props = context.scene.sinter_forge_tool

        # Presets
        box = layout.box()
        row = box.row()
        row.label(text="Library")
        row.operator("sf.save_preset", text="", icon='ADD')
        
        preset_dir = os.path.join(os.path.dirname(__file__), "presets")
        if os.path.exists(preset_dir):
            col = box.column(align=True)
            for f_path in glob.glob(os.path.join(preset_dir, "*.json")):
                f_name = os.path.basename(f_path)
                try:
                    with open(f_path, 'r') as f:
                        data = json.load(f)
                        btn_name = data.get("name", f_name)
                except:
                    btn_name = f_name
                op = col.operator("sf.load_json_preset", text=btn_name)
                op.filename = f_name
        else:
            box.label(text="No presets folder found!", icon='ERROR')

        layout.separator()
        layout.label(text="Advanced Settings", icon='MODIFIER')
        layout.prop(props, "method", text="")
        layout.prop(props, "fix_scale")
        
        box = layout.box()
        box.label(text="Surface Texture")
        box.prop(props, "texture_type", text="")
        col = box.column(align=True)
        col.prop(props, "strength")
        col.prop(props, "noise_scale")
        col.prop(props, "noise_detail") 

        box = layout.box()
        box.label(text="Resolution & Optimization")
        if props.method == 'MECHANICAL':
            box.prop(props, "subdiv_levels")
        else:
            box.prop(props, "voxel_size")
        
        row = box.row()
        row.prop(props, "use_decimate")
        if props.use_decimate:
            row.prop(props, "decimate_ratio")

        layout.separator()
        row = layout.row()
        row.scale_y = 2.0
        row.operator("sf.apply_texture", icon='SHADING_RENDERED')

        # --- SPONSOR FOOTER ---
        layout.separator()
        row = layout.row()
        row.alignment = 'CENTER'
        op = row.operator("wm.url_open", text="Made possible by AUS GmbH", icon='HEART')
        op.url = "https://aus.gmbh"

# --- REGISTRATION ---

classes = (
    SinterForgePreferences,
    SinterForgeSettings,
    SF_OT_ApplyTexture,
    SF_OT_SavePreset,
    SF_OT_LoadJSONPreset,
    SF_PT_MainPanel,
)

def register():
    for cls in classes:
        bpy.utils.register_class(cls)
    bpy.types.Scene.sinter_forge_tool = bpy.props.PointerProperty(type=SinterForgeSettings)

def unregister():
    for cls in reversed(classes):
        bpy.utils.unregister_class(cls)
    del bpy.types.Scene.sinter_forge_tool

if __name__ == "__main__":
    register()